import Vue from "vue";
import moment from "moment";
import isDate from "lodash/isDate";
import isString from "lodash/isString";
import Pikaday from "pikaday";

export default Vue.extend({

    data() {

        return {
            buyInsLabels: {},
            typesLabels: {},
            isExpanded: false,
            selectedMenu: undefined,
            selectedBuyIns: [],
            selectedBuyInsData: [],
            selectedTypes: [],
            selectedTypesData: [],
            startDate: undefined,
            endDate: undefined
        }
    },
    watch: {

        selectedBuyIns(value)
        {
            this.selectedBuyInsData = value.map(buyInValue => ({
                label: this.buyInsLabels[buyInValue],
                value: buyInValue
            }));
        },
        selectedTypes(value)
        {
            this.selectedTypesData = value.map(typeValue => ({
                label: this.typesLabels[typeValue],
                value: typeValue
            }));
        }
    },
    mounted()
    {
        /*
         * Choices labels fetching
         */
        const buyInsLabels = this.$el.querySelectorAll('.c-tournaments-filter-checkbox-ui-label[data-type="buyIn"]');
        const typesLabels = this.$el.querySelectorAll('.c-tournaments-filter-checkbox-ui-label[data-type="type"]');

        for (let buyInsLabel of buyInsLabels)
        {
            this.buyInsLabels[buyInsLabel.dataset.value] = buyInsLabel.textContent;
        }
        for (let typeLabel of typesLabels)
        {
            this.typesLabels[typeLabel.dataset.value] = typeLabel.textContent;
        }

        /*
         * DatePickers init
         */
        const commonOptions = {
            showDaysInNextAndPreviousMonths: true,
            enableSelectionDaysInNextAndPreviousMonths: true,
            firstDay: 1,
            i18n: {
                months: window.tabMonthsFull,
                weekdays: window.tabDay,
                weekdaysShort: window.tabDayShort
            },
            minDate: new Date(),
            theme: "pikaday-barriere-light"
        };

        const startDatePickerOptions = Object.assign({}, commonOptions, {
            container: this.$refs.startDatePicker,
            field: this.$refs.startDatePickerInput,
            onSelect: date => this.startDate = moment(date).format("YYYY-MM-DD")
        });

        const endDatePickerOptions = Object.assign({}, commonOptions, {
            container: this.$refs.endDatePicker,
            field: this.$refs.endDatePickerInput,
            onSelect: date => this.endDate = moment(date).format("YYYY-MM-DD")
        });

        new Pikaday(startDatePickerOptions);
        new Pikaday(endDatePickerOptions);
    },
    computed: {

        isStartDateSelected()
        {
            return isString(this.startDate) && this.startDate.length > 0;
        },
        isEndDateSelected()
        {
            return isString(this.endDate) && this.endDate.length > 0;
        },
        isMenuExpanded()
        {
            return this.selectedMenu !== undefined;
        }
    },
    filters: {

        getDayOfWeek(dateString)
        {
            return moment(dateString).format("dddd");
        },
        getFormattedDate(dateString)
        {
            return moment(dateString).format("LL");
        }
    },
    methods: {

        isMenuSelected(name)
        {
            return this.selectedMenu === name;
        },
        selectMenu(name)
        {
            this.selectedMenu = this.selectedMenu === name ? undefined : name;
        },
        toggle()
        {
            this.isExpanded = !this.isExpanded;
        },
        unSelectBuyIn(buyIn)
        {
            this.selectedBuyIns = this.selectedBuyIns.filter(value => value !== buyIn.value);
        },
        unSelectStartDate()
        {
            this.startDate = undefined;
        },
        unSelectEndDate()
        {
            this.endDate = undefined;
        },
        unSelectType(type)
        {
            this.selectedTypes = this.selectedTypes.filter(value => value !== type.value);
        }
    }
});
